/*
 * EffectInfo.h
 *
 * Created 8/30/2009 By Johnny Huynh
 *
 * Version 00.00.01 8/30/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 /**
  * Effects apply to targets that are hit by a DamageObject
  */
 
 #ifndef EFFECT_INFO_H
 #define EFFECT_INFO_H
 
 template <typename T> class EffectInfo;
 
 #include "global.h"
 
 #include "Object.h"
 #include "referenceCount.h"
 #include "pointerTo.h"
 
 /**
  * Class specification for EffectInfo
  */
 template <typename T>
 class EffectInfo : public ReferenceCount
 {
 public:
    typedef void EffectFunc( EffectInfo<T>* effect_info_Ptr, Object<T>* into_object_Ptr, const CollisionEntry& c_entry );
 
 // Data Members
 private:
    EffectFunc* _effect_func;
    
 // Local Functions
 public:
    EffectInfo( EffectFunc* effect_func );
    EffectInfo( const EffectInfo<T>& effect_info );
    virtual ~EffectInfo();
    inline EffectInfo<T>& operator=( const EffectInfo<T>& effect_info );
    virtual inline void apply_effect( Object<T>& object, const CollisionEntry& c_entry );
 
 // Protected Functions
 protected:
    inline void set_effect_func( EffectFunc* effect_func );
 
 // Private Functions
 private:
    
 // Friend Functions
 public:
    
 };
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 EffectInfo<T>::EffectInfo( EffectFunc* effect_func )
               : ReferenceCount(),
                 _effect_func( effect_func )
 {
    nassertv( effect_func != NULL );
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 EffectInfo<T>::EffectInfo( const EffectInfo<T>& effect_info )
               : ReferenceCount( effect_info ),
                 _effect_func( effect_info._effect_func )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 EffectInfo<T>::~EffectInfo()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified EffectInfo to this EffectInfo.
  *
  * @param (const EffectInfo<T>&) effect_info
  * @return EffectInfo<T>&
  */
 template <typename T>
 inline EffectInfo<T>& EffectInfo<T>::operator=( const EffectInfo<T>& effect_info )
 {
    ReferenceCount::operator=( effect_info );
    _effect_func = effect_info._effect_func;
    
    return *this;
 }
 
 /**
  * apply_effect() applies the effect (specified by this EffectInfo) onto the specified Object.
  *
  * @param (Object<T>*) object
  * @param (const CollisionEntry&) c_entry
  */
 template <typename T>
 inline void EffectInfo<T>::apply_effect( Object<T>& object, const CollisionEntry& c_entry )
 {
    (*_effect_func)( this, &object, c_entry );
 }
 
 /** PROTECTED FUNCTIONS **/
 
 /**
  * set_effect_func() sets the effect function to the specified the EffectFunc.
  *
  * @param (EffectFunc*) effect_func
  */
 template <typename T>
 inline void EffectInfo<T>::set_effect_func( EffectFunc* effect_func )
 {
    nassertv( effect_func != NULL );
    
    _effect_func = effect_func;
 }
 
 /** FRIEND FUNCTIONS **/
 
 #endif // EFFECT_INFO_H